//
//  TWEnlargedStreamTableViewCell.m
//  Twitch
//
//  Created by Auston Stewart on 11/25/13.
//  Copyright (c) 2013 Justin.tv. All rights reserved.
//

#import "TWEnlargedStreamTableViewCell.h"
#import <QuartzCore/QuartzCore.h>
#import "UIFont+Twitch.h"
#import "NSString+Twitch.h"

NSString *const TWEnlargedStreamTableViewCellReuseIdentifier = @"TWEnlargedStreamTableViewCellReuseIdentifier";

@implementation TWEnlargedStreamTableViewCell
{
	CAGradientLayer *_gradientOverlay;
	UILabel *_channelNameLabel;
	UILabel *_broadcastTitleLabel;
	UILabel *_broadcastInfoLabel;
	
	CGFloat _channelNameOffset;
	CGFloat _broadcastTitleOffset;
	CGFloat _viewerCountOffset;
	
	NSString *_channelDisplayName;
	NSString *_gameName;
	NSString *_broadcastTitle;
	NSUInteger _viewerCount;
	
	BOOL _viewerCountIsSet;
}

#define TWEnlargedStreamPreviewImageWidth 300.f
#define TWEnlargedStreamPreviewImageHeight 169.f
#define TWEnlargedStreamSpacing 10.f
#define TWEnlargedStreamCellHeight TWEnlargedStreamPreviewImageHeight
#define TWEnlargedStreamBroadcasterLabelOffsetWithoutTitle 126.f
#define TWEnlargedStreamBroadcasterLabelOffsetWithTitle 109.f
#define TWEnlargedStreamBroadcastTitleOffset 130.f
#define TWEnlargedStreamBroadcastTitleLabelHeight 14.f
#define TWEnlargedStreamBroadcasterLabelHeight 19.f
#define TWEnlargedStreamViewerCountLabelOffset 147.f
#define TWEnlargedStreamViewerCountLabelHeight 14.f

+ (NSString *)reuseIdentifier
{
	return TWEnlargedStreamTableViewCellReuseIdentifier;
}

+ (CGFloat)preferredHeight
{
	return TWEnlargedStreamCellHeight;
}

- (void)commonInitialization
{
	[super commonInitialization];
	
	self.contentVerticalAlignment = TWTableViewCellContentVerticalAlignmentCenter;
	self.imageSize = CGSizeMake(TWEnlargedStreamPreviewImageWidth, TWEnlargedStreamPreviewImageHeight);
	self.imageView.contentMode = UIViewContentModeScaleAspectFill;
	self.placeholderImageName = @"404_channel_iphone_grid";
	self.accessoryView = nil;
	self.selectedBackgroundView = nil;
	self.selectionStyle = UITableViewCellSelectionStyleNone;
	
	CALayer *enlargedStreamGradientOverlay = [[CALayer alloc] init];
	UIImage *gradientOverlayImage = [UIImage imageFromFrameworkBundleNamed:@"gradient_channel_iphone_grid"];
	enlargedStreamGradientOverlay.contents = (__bridge id)gradientOverlayImage.CGImage;
	enlargedStreamGradientOverlay.frame = CGRectMake(0.f, 0.f, TWEnlargedStreamPreviewImageWidth, TWEnlargedStreamPreviewImageHeight);
	self.imageView.layer.masksToBounds = YES;
	[self.imageView.layer addSublayer:enlargedStreamGradientOverlay];
	
	// Create custom labels
	_channelNameLabel = [[UILabel alloc] initWithFrame:CGRectMake(TWEnlargedStreamSpacing * 2.f, 0.f, TWEnlargedStreamPreviewImageWidth - (TWEnlargedStreamSpacing * 2.f), TWEnlargedStreamBroadcasterLabelHeight)];
	_channelNameLabel.font = [UIFont twitchEnlargedLiveChannelCellBroadcasterFont];
	_channelNameLabel.textColor = [UIColor twitchWhiteOverlayTextColor];
	
	_broadcastInfoLabel = [[UILabel alloc] initWithFrame:CGRectMake(TWEnlargedStreamSpacing * 2.f, 0.f, TWEnlargedStreamPreviewImageWidth - (TWEnlargedStreamSpacing * 2.f), TWEnlargedStreamViewerCountLabelHeight)];
	_broadcastInfoLabel.font = [UIFont twitchEnlargedLiveChannelCellViewerCountFont];
	_broadcastInfoLabel.textColor = [UIColor twitchPurpleOverlayTextColor];
	
	_broadcastTitleLabel = [[UILabel alloc] initWithFrame:CGRectMake(TWEnlargedStreamSpacing * 2.f, 0.f, TWEnlargedStreamPreviewImageWidth - TWEnlargedStreamSpacing - floorf(TWEnlargedStreamPreviewImageWidth / 3.f), TWEnlargedStreamBroadcastTitleLabelHeight)];
	_broadcastTitleLabel.font = [UIFont twitchCellBroadcastTitleFont];
	_broadcastTitleLabel.textColor = [UIColor twitchGridItemBroadcastTitleColor];
	_broadcastTitleLabel.textAlignment = NSTextAlignmentLeft;
	_broadcastTitleLabel.lineBreakMode = NSLineBreakByTruncatingTail;
	_broadcastTitleLabel.hidden = YES;
	
	[self.contentView addSubview:_channelNameLabel];
	[self.contentView addSubview:_broadcastTitleLabel];
	[self.contentView addSubview:_broadcastInfoLabel];
}

- (void)prepareForReuse
{
	[super prepareForReuse];
	
	_viewerCountIsSet = NO;
	_gameName = nil;
	
	// Hide the broadcast title
	_broadcastTitleLabel.hidden = YES;
	self.contentVerticalAlignment = TWTableViewCellContentVerticalAlignmentCenter;
}

- (void)setChannelDisplayName:(NSString *)channelDisplayName
{
	_channelDisplayName = [channelDisplayName copy];
	_channelNameLabel.text = channelDisplayName;
}

- (void)updateCombinedDetailTextLabel
{
	NSString *viewersString = [@(_viewerCount) formattedLocalizedNumericRepresentation];
	NSString *localizedCombinedDetailTextString = TKLocalizedString(@"playing %@ for %@ viewers",@"Enlarged stream cell combined detail text");
	NSRange gameNameSpecifierRange = [localizedCombinedDetailTextString tw_rangeOfStringSpecifier:1];
	NSRange viewerCountSpecifierRange = [localizedCombinedDetailTextString tw_rangeOfStringSpecifier:2];
	
	NSMutableAttributedString *attributedDetailText = [[NSMutableAttributedString alloc] initWithString:[NSString		stringWithFormat:TKLocalizedString(@"playing %@ for %@ viewers",@"Enlarged stream cell combined detail text"),_gameName,viewersString]];
	
	if (gameNameSpecifierRange.location != NSNotFound && viewerCountSpecifierRange.location != NSNotFound) {
		
		if (gameNameSpecifierRange.location > viewerCountSpecifierRange.location) {
			
			gameNameSpecifierRange.location += viewersString.length - gameNameSpecifierRange.length;
		}
		else {
			
			viewerCountSpecifierRange.location += _gameName.length - gameNameSpecifierRange.length;
		}
		
		[attributedDetailText addAttribute:NSFontAttributeName value:[UIFont twitchEnlargedLiveChannelCellViewerCountBoldFont] range:NSMakeRange(gameNameSpecifierRange.location,_gameName.length)];
		[attributedDetailText addAttribute:NSFontAttributeName value:[UIFont twitchEnlargedLiveChannelCellViewerCountBoldFont] range:NSMakeRange(viewerCountSpecifierRange.location,viewersString.length)];
	}
	_broadcastInfoLabel.attributedText = attributedDetailText;
}

- (void)setViewerCount:(NSInteger)viewerCount
{
	_viewerCount = viewerCount;
	_viewerCountIsSet = YES;
	
	if (_gameName) [self updateCombinedDetailTextLabel];
	else {
		
		NSString *viewersString = [@(_viewerCount) formattedLocalizedNumericRepresentation];
		NSString *localizedViewerCountString = TKLocalizedString(@"%@ viewers", @"Enlarged stream cell viewer count text");
		NSRange viewCountSpecifierRange = [localizedViewerCountString tw_rangeOfStringSpecifier:1];
		NSMutableAttributedString *attributedViewerCount = [[NSMutableAttributedString alloc] initWithString:[NSString stringWithFormat:TKLocalizedString(@"%@ viewers", @"Enlarged stream cell viewer count text"),viewersString]];
		
		if (viewCountSpecifierRange.location != NSNotFound) {
			[attributedViewerCount addAttribute:NSFontAttributeName value:[UIFont twitchEnlargedLiveChannelCellViewerCountBoldFont] range:NSMakeRange(viewCountSpecifierRange.location,viewersString.length)];
		}
		_broadcastInfoLabel.attributedText = attributedViewerCount;
	}
}

- (void)setGameName:(NSString *)gameName
{
	_gameName = [gameName copy];
	
	if (_viewerCountIsSet) [self updateCombinedDetailTextLabel];
	else {
		
		NSString *localizedGameNameString = TKLocalizedString(@"playing %@",@"Enlarged stream cell game name text");
		NSRange gameNameSpecifierRange = [localizedGameNameString tw_rangeOfStringSpecifier:1];
		NSMutableAttributedString *attributedGameName = [[NSMutableAttributedString alloc] initWithString:[NSString		stringWithFormat:TKLocalizedString(@"playing %@",@"Enlarged stream cell game name text"),_gameName]];
		
		if (gameNameSpecifierRange.location != NSNotFound) {
			[attributedGameName addAttribute:NSFontAttributeName value:[UIFont twitchEnlargedLiveChannelCellViewerCountBoldFont] range:NSMakeRange(gameNameSpecifierRange.location,_gameName.length)];
		}
		_broadcastInfoLabel.attributedText = attributedGameName;
	}
}

- (void)setBroadcastTitle:(NSString *)broadcastTitle
{
	if (broadcastTitle.length) {
		
		_broadcastTitle = [broadcastTitle copy];
		_broadcastTitleLabel.text = _broadcastTitle;
		
		// Make space for the broadcast title
		_broadcastTitleLabel.hidden = NO;
		[self setNeedsLayout];
	}
}

- (void)adjustForHighlightState:(BOOL)highlighted
{
	CGPoint broadcasterLabelOrigin = CGPointMake(TWEnlargedStreamSpacing * 2.f, _channelNameOffset);
	CGPoint broadcastTitleOrigin = CGPointMake(TWEnlargedStreamSpacing * 2.f, _broadcastTitleOffset);
	CGPoint viewerCountLabelOrigin = CGPointMake(TWEnlargedStreamSpacing * 2.f, _viewerCountOffset);
	CGRect broadcasterLabelFrame = _channelNameLabel.frame;
	CGRect broadcastTitleFrame = _broadcastTitleLabel.frame;
	CGRect viewerCountLabelFrame = _broadcastInfoLabel.frame;
	broadcasterLabelFrame.origin = broadcasterLabelOrigin;
	broadcastTitleFrame.origin = broadcastTitleOrigin;
	viewerCountLabelFrame.origin = viewerCountLabelOrigin;
	CGRect bounds = self.imageView.bounds;
	bounds.size = CGSizeMake(TWEnlargedStreamPreviewImageWidth,TWEnlargedStreamPreviewImageHeight);
	if (highlighted) {
		
		CGSize initialSize = bounds.size;
		
		// Determine new image bounds
		bounds.size.width = floorf(.95 * bounds.size.width);
		bounds.size.height = floorf(.95 * bounds.size.height);
		
		// Determine new text frames
		CGFloat xDelta = floorf((initialSize.width - bounds.size.width) / 2.f);
		CGFloat yDelta = floorf((bounds.size.height - initialSize.height) / 2.f);
		broadcasterLabelFrame.origin.x += xDelta;
		broadcasterLabelFrame.origin.y += yDelta;
		broadcastTitleFrame.origin.x += xDelta;
		broadcastTitleFrame.origin.y += yDelta;
		viewerCountLabelFrame.origin.x += xDelta;
		viewerCountLabelFrame.origin.y += yDelta;
	}
	self.imageView.bounds = bounds;
	_channelNameLabel.frame = broadcasterLabelFrame;
	_broadcastTitleLabel.frame = broadcastTitleFrame;
	_broadcastInfoLabel.frame = viewerCountLabelFrame;
}

- (void)setHighlighted:(BOOL)highlighted
{
	[super setHighlighted:highlighted];
	[self adjustForHighlightState:highlighted];
}

- (void)setHighlighted:(BOOL)highlighted animated:(BOOL)animated
{
	[super setHighlighted:highlighted animated:animated];
	// Just tapping will not cause the animated BOOL to be set, so animate regardless
	[UIView animateWithDuration:.25 animations:^{
		
		[self adjustForHighlightState:highlighted];
	}];
}

// HACK: This is all a crazy hack to allow for UICollectionView grid-esque spacing of these enlarged previews on iPhone.
// It's necessary because contentInsets on UITableView apply to the section header rather than the content and there's no item spacing.
// We should really just reimplement our table view styles using UICollectionViewCells, as crazy as that sounds.
- (void)layoutSubviews
{
	CGFloat offsetIncrement = 0.f;
	
	[super layoutSubviews];
	
	_channelNameOffset = _broadcastTitleLabel.hidden ? TWEnlargedStreamBroadcasterLabelOffsetWithoutTitle : TWEnlargedStreamBroadcasterLabelOffsetWithTitle;
	_broadcastTitleOffset = TWEnlargedStreamBroadcastTitleOffset;
	_viewerCountOffset = TWEnlargedStreamViewerCountLabelOffset;
	
	if (self.contentVerticalAlignment == TWTableViewCellContentVerticalAlignmentCenter) {
		
		offsetIncrement = floorf((self.bounds.size.height - [[self class] preferredHeight]) / 2.f);
	}
	else if (self.contentVerticalAlignment == TWTableViewCellContentVerticalAlignmentBottom) {
		
		offsetIncrement = (self.bounds.size.height - [[self class] preferredHeight]);
	}
	_channelNameOffset += offsetIncrement;
	_broadcastTitleOffset += offsetIncrement;
	_viewerCountOffset += offsetIncrement;
	
	CGRect frame = _channelNameLabel.frame;
	frame.origin.y = _channelNameOffset;
	_channelNameLabel.frame = frame;
	frame = _broadcastTitleLabel.frame;
	frame.origin.y = _broadcastTitleOffset;
	_broadcastTitleLabel.frame = frame;
	frame = _broadcastInfoLabel.frame;
	frame.origin.y = _viewerCountOffset;
	_broadcastInfoLabel.frame = frame;
}

#undef TWEnlargedStreamPreviewImageWidth
#undef TWEnlargedStreamPreviewImageHeight
#undef TWEnlargedStreamSpacing
#undef TWEnlargedStreamCellHeight
#undef TWEnlargedStreamBroadcasterLabelOffset
#undef TWEnlargedStreamViewerCountLabelOffset
#undef TWEnlargedStreamBroadcasterLabelHeight
#undef TWEnlargedStreamViewerCountLabelHeight

@end
